/*:
 * @target MZ
 * @plugindesc v1.3 イベントを放物線で吹っ飛ばし、回転/拡縮、画面外で消滅（erase）する
 * @author あん
 *
 * @help BlowAwayEventMZ.js
 * 使い方：
 *  1) このプラグインを有効化
 *  2) イベントの「プラグインコマンド」→ 本プラグインを選び
 *     コマンド「吹っ飛ばす」を置くだけ（対象＝このイベント）
 *
 * 備考：
 *  ・飛行中は through=true。見た目のみ移動し、画面外で erase()。
 *  ・コメントタグ <BlowAway auto ...> でも自動発動できます（任意）。
 *
 * @command BlowAway
 * @text 吹っ飛ばす
 * @desc 対象イベントを放物線で吹っ飛ばし、画面外で消滅させる
 *
 * @arg target
 * @text 対象
 * @type select
 * @option このイベント @value this
 * @option 指定ID       @value id
 * @default this
 *
 * @arg eventId
 * @text イベントID（target=id時のみ）
 * @type number
 * @min 1
 * @default 1
 *
 * @arg direction
 * @text 方向
 * @type select
 * @option プレイヤー向き @value player
 * @option 上 @value up
 * @option 下 @value down
 * @option 左 @value left
 * @option 右 @value right
 * @default player
 *
 * @arg speed
 * @text 水平スピード（px/フレ）
 * @type number
 * @decimals 2
 * @min 0
 * @default 5
 *
 * @arg jump
 * @text 垂直初速（上向き, px/フレ）
 * @type number
 * @decimals 2
 * @min 0
 * @default 9
 *
 * @arg gravity
 * @text 重力（px/フレ^2）
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.5
 *
 * @arg spin
 * @text 回転速度（rad/フレ）
 * @type number
 * @decimals 3
 * @min 0
 * @default 0.4
 *
 * @arg scaleStart
 * @text スケール開始
 * @type number
 * @decimals 2
 * @min 0.1
 * @default 1.00
 *
 * @arg scaleEnd
 * @text スケール終了
 * @type number
 * @decimals 2
 * @min 0.1
 * @default 0.80
 *
 * @arg animHit
 * @text 開始アニメID（0で無し）
 * @type animation
 * @default 0
 *
 * @arg animEnd
 * @text 終了アニメID（0で無し）
 * @type animation
 * @default 0
 *
 * @arg offscreenMargin
 * @text 画面外判定マージン(px)
 * @type number
 * @min 0
 * @default 96
 */
(() => {
  const PLUGIN_NAME = (() => {
    const s = document.currentScript && document.currentScript.src;
    const m = s && s.match(/([^\/\\]+)\.js$/i);
    return m ? m[1] : "BlowAwayEventMZ";
  })();

  const dirFromPlayer = () => {
    const d = $gamePlayer.direction();
    return d === 2 ? "down" : d === 4 ? "left" : d === 6 ? "right" : "up";
  };

  // ========= Event runtime =========
  Game_Event.prototype.startBlowAway = function(p={}) {
    if (this._blowActive) return;
    const opt = Object.assign({
      direction: "player",
      speed: 5, jump: 9, gravity: 0.5,
      spin: 0.4, scaleStart: 1.0, scaleEnd: 0.8,
      animHit: 0, animEnd: 0, offscreenMargin: 96
    }, p);

    const dir = opt.direction === "player" ? dirFromPlayer() : opt.direction;
    const vx = dir === "left" ? -opt.speed : dir === "right" ? opt.speed : 0;
    const vy = -opt.jump + (dir === "up" ? -opt.speed*0.25 : dir === "down" ? opt.speed*0.25 : 0);

    this._blow = {
      vx, vy, gravity: opt.gravity,
      spin: opt.spin, scaleStart: opt.scaleStart, scaleEnd: opt.scaleEnd,
      t: 0, animEnd: opt.animEnd, margin: opt.offscreenMargin
    };
    this._blowOffX = 0; this._blowOffY = 0; this._blowRot = 0;
    this._blowScale = opt.scaleStart; this._blowActive = true;

    this._throughMemo = this.isThrough();
    this.setThrough(true);

    if (opt.animHit > 0) $gameTemp.requestAnimation([this], opt.animHit);
  };

  Game_Event.prototype.isBlowing = function(){ return !!this._blowActive; };

// ===== 差し替え開始 =====
const _GE_update = Game_Event.prototype.update;
Game_Event.prototype.update = function() {
  _GE_update.call(this);
  if (this._blowActive) this.updateBlow();
};

Game_Event.prototype.updateBlow = function() {
  const b = this._blow;

  // まだ“画面外に出ていない”通常飛行中は物理更新
  if (!b.ending) {
    this._blowOffX += b.vx;
    this._blowOffY += b.vy;
    b.vy += b.gravity;

    this._blowRot   += b.spin;
    this._blowScale  = b.scaleStart + (b.scaleEnd - b.scaleStart) * Math.min(1, b.t/60);
    b.t++;

    // 画面外判定
    const sx = this.screenX() + (this._blowOffX || 0);
    const sy = this.screenY() + (this._blowOffY || 0);
    const m = b.margin || 0;
    const out = sx < -m || sx > Graphics.width + m || sy < -m || sy > Graphics.height + m;

    if (out || b.t > 600) {
      // ★ 終了フェーズへ移行：この時点の見た目座標を保ったままアニメを再生
      b.ending = true;
      b.endWait = 90;          // アニメを見せる猶予（必要に応じて調整）
      b.vx = 0; b.vy = 0;      // 位置固定
      b.spin = 0;              // 回転停止（続けたいならこの行を消す）

      if (b.animEnd > 0) {
        $gameTemp.requestAnimation([this], b.animEnd); // ← 固定座標のまま再生
      }
    }
    return;
  }

  // 終了フェーズ：固定表示のまま少し待ってから消去
  if (b.endWait > 0) {
    b.endWait--;
    return;
  }

  // 片付け
  this.setThrough(!!this._throughMemo);
  this._blowActive = false;
  this.erase();
};
// ===== 差し替え終わり =====

// ========= Sprite side (apply visual offsets safely) =========
const _SC_update = Sprite_Character.prototype.update;
Sprite_Character.prototype.update = function() {
  // まず標準更新で今フレームの“基準座標”を確定
  _SC_update.call(this);

  const ev = this._character;
  if (ev && ev.isBlowing && ev.isBlowing()) {
    // 標準更新結果を基準として、毎フレームオフセットを加算
    const baseX = this.x;
    const baseY = this.y;

    this.rotation = ev._blowRot || 0;
    this.scale.x = this.scale.y = ev._blowScale || 1;

    this.x = baseX + (ev._blowOffX || 0);
    this.y = baseY + (ev._blowOffY || 0);
  } else {
    // 非吹っ飛び中は変形だけリセット（位置は標準更新に任せる）
    this.rotation = 0;
    this.scale.x = this.scale.y = 1;
  }
};

  // ========= Optional: comment tag auto =========
  const parseTag = (text) => {
    const m = /<\s*BlowAway\b([^>]*)>/i.exec(text || "");
    if (!m) return null;
    const kv = m[1] || ""; const opt = {auto:/\bauto\b/i.test(kv)};
    const pick = (k,f=(v)=>v)=>{ const r=new RegExp(`${k}\\s*=\\s*([^\\s>]+)`,"i").exec(kv); if(r) opt[k]=f(r[1]); };
    pick("speed", parseFloat); pick("jump", parseFloat); pick("gravity", parseFloat);
    pick("spin", parseFloat); pick("animHit", Number); pick("animEnd", Number); pick("offscreenMargin", Number);
    const sm = /scale\s*=\s*([0-9.]+)\s*->\s*([0-9.]+)/i.exec(kv);
    if (sm) { opt.scaleStart=parseFloat(sm[1]); opt.scaleEnd=parseFloat(sm[2]); }
    return opt;
  };

  const _GE_setupPage = Game_Event.prototype.setupPage;
  Game_Event.prototype.setupPage = function() {
    _GE_setupPage.call(this);
    this._blowAutoOpt = null;
    if (!this.page()) return;
    let c=""; for (const cmd of this.list()||[]) if (cmd.code===108||cmd.code===408) c+=cmd.parameters[0]+"\n";
    const opt = parseTag(c);
    if (opt) this._blowAutoOpt = Object.assign({
      speed:5,jump:9,gravity:0.5,spin:0.4,scaleStart:1,scaleEnd:0.8,animHit:0,animEnd:0,offscreenMargin:96
    }, opt);
  };

  const _GP_startMapEvent = Game_Player.prototype.startMapEvent;
  Game_Player.prototype.startMapEvent = function(x,y,tr,normal){
    const evs=$gameMap.eventsXy(x,y);
    for(const ev of evs){
      if(ev && ev._blowAutoOpt && ev._blowAutoOpt.auto && !ev.isBlowing()){
        ev.startBlowAway(Object.assign({direction:"player"}, ev._blowAutoOpt));
      }
    }
    _GP_startMapEvent.call(this,x,y,tr,normal);
  };

  // ========= Plugin Command =========
  const register = (name)=>{
    PluginManager.registerCommand(name,"BlowAway",args=>{
      const target=args.target||"this"; let ev=null;
      if(target==="this"){
        const eid=$gameMap?._interpreter?.eventId?.()||0;
        if(eid>0){ ev=$gameMap.event(eid); }
        else {
          const px=$gamePlayer.x, py=$gamePlayer.y, d=$gamePlayer.direction();
          const dx=d===6?1:d===4?-1:0, dy=d===2?1:d===8?-1:0;
          ev=$gameMap.eventsXy(px,py)[0]||$gameMap.eventsXy(px+dx,py+dy)[0]||null;
        }
      }else{
        ev=$gameMap.event(Number(args.eventId||0));
      }
      if(!ev) return;
      ev.startBlowAway({
        direction: args.direction||"player",
        speed: Number(args.speed||5),
        jump: Number(args.jump||9),
        gravity: Number(args.gravity||0.5),
        spin: Number(args.spin||0.4),
        scaleStart: Number(args.scaleStart||1),
        scaleEnd: Number(args.scaleEnd||0.8),
        animHit: Number(args.animHit||0),
        animEnd: Number(args.animEnd||0),
        offscreenMargin: Number(args.offscreenMargin||96)
      });
    });
  };
  register(PLUGIN_NAME);
  if(PLUGIN_NAME!=="BlowAwayEventMZ") register("BlowAwayEventMZ");
})();
